<?php
/*
 * Elementor Apeix Testimonial Widget
 * Author & Copyright: wpoceans
*/

namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

class Apeix_Testimonial extends Widget_Base
{

	/**
	 * Retrieve the widget name.
	 */
	public function get_name()
	{
		return 'tmx-apeix_testimonial';
	}

	/**
	 * Retrieve the widget title.
	 */
	public function get_title()
	{
		return esc_html__('Testimonial', 'apeix-core');
	}

	/**
	 * Retrieve the widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-testimonial';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 */
	public function get_categories()
	{
		return ['wpoceans-category'];
	}

	/**
	 * Retrieve the list of scripts the Apeix Testimonial widget depended on.
	 * Used to set scripts dependencies required to run the widget.
	 */
	public function get_script_depends()
	{
		return ['tmx-apeix_testimonial'];
	}

	/**
	 * Register Apeix Testimonial widget controls.
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 */
	protected function register_controls()
	{

		$this->start_controls_section(
			'section_testimonial',
			[
				'label' => esc_html__('Testimonial Options', 'apeix-core'),
			]
		);
		$repeater = new Repeater();
		$repeater->add_control(
			'testimonial_title',
			[
				'label' => esc_html__('Testimonial Title Text', 'apeix-core'),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__('Title Text', 'apeix-core'),
				'placeholder' => esc_html__('Type title text here', 'apeix-core'),
				'label_block' => true,
			]
		);
		$repeater->add_control(
			'testimonial_subtitle',
			[
				'label' => esc_html__('Testimonial Sub Title', 'apeix-core'),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__('Testimonial Sub Title', 'apeix-core'),
				'placeholder' => esc_html__('Type testimonial Sub title here', 'apeix-core'),
				'label_block' => true,
			]
		);
		$repeater->add_control(
			'testimonial_content',
			[
				'label' => esc_html__('Testimonial Content', 'apeix-core'),
				'type' => Controls_Manager::TEXTAREA,
				'default' => esc_html__('Testimonial Content', 'apeix-core'),
				'placeholder' => esc_html__('Type testimonial Content here', 'apeix-core'),
				'label_block' => true,
			]
		);
		$repeater->add_control(
			'bg_image',
			[
				'label' => esc_html__('Testimonial Image', 'apeix-core'),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],

			]
		);
		$repeater->add_control(
			'rating_image',
			[
				'label' => esc_html__('Rating Image', 'apeix-core'),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],

			]
		);
		$this->add_control(
			'testimonialItems_groups',
			[
				'label' => esc_html__('Testimonial Items', 'apeix-core'),
				'type' => Controls_Manager::REPEATER,
				'default' => [
					[
						'testimonial_title' => esc_html__('Testimonial', 'apeix-core'),
					],

				],
				'fields' =>  $repeater->get_controls(),
				'title_field' => '{{{ testimonial_title }}}',
			]
		);
		$this->end_controls_section(); // end: Section


		// Title
		$this->start_controls_section(
			'section_title_style',
			[
				'label' => esc_html__('Title', 'apeix-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'apeix-core'),
				'name' => 'apeix_title_typography',
				'selector' => '{{WRAPPER}} .wpo-testimonial-item h4',
			]
		);
		$this->add_control(
			'title_color',
			[
				'label' => esc_html__('Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .wpo-testimonial-item h4' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'title_padding',
			[
				'label' => __('Title Padding', 'apeix-core'),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em'],
				'selectors' => [
					'{{WRAPPER}} .wpo-testimonial-item h4' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section

		// SubTitle
		$this->start_controls_section(
			'section_sutitle_style',
			[
				'label' => esc_html__('SubTitle', 'apeix-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'apeix-core'),
				'name' => 'apeix_subtitle_typography',
				'selector' => '{{WRAPPER}} .wpo-testimonial-item span',
			]
		);
		$this->add_control(
			'sutitle_color',
			[
				'label' => esc_html__('Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .wpo-testimonial-item span' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'sutitle_padding',
			[
				'label' => esc_html__('Title Padding', 'apeix-core'),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em'],
				'selectors' => [
					'{{WRAPPER}}  .wpo-testimonial-item span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section


		// Content
		$this->start_controls_section(
			'section_content_style',
			[
				'label' => esc_html__('Content', 'apeix-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'apeix-core'),
				'name' => 'ntrsv_content_typography',
				'selector' => '{{WRAPPER}} .wpo-testimonial-item p',
			]
		);
		$this->add_control(
			'content_color',
			[
				'label' => esc_html__('Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .wpo-testimonial-item p' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'content_padding',
			[
				'label' => esc_html__('Content Padding', 'apeix-core'),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em'],
				'selectors' => [
					'{{WRAPPER}} .wpo-testimonial-item p' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section


		// Dots
		$this->start_controls_section(
			'section_dot_style',
			[
				'label' => esc_html__('Dots', 'apeix-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'dot_color',
			[
				'label' => esc_html__('Dot Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .owl-theme .owl-dots .owl-dot span' => 'background-color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'dot_active_color',
			[
				'label' => esc_html__('Active Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .owl-theme .owl-dots .owl-dot.active span' => 'background-color: {{VALUE}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section


	}

	/**
	 * Render Testimonial widget output on the frontend.
	 * Written in PHP and used to generate the final HTML.
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();
		$testimonialItems_groups = !empty($settings['testimonialItems_groups']) ? $settings['testimonialItems_groups'] : [];

		// Turn output buffer on
		ob_start(); ?>
		<div class="wpo-testimonial-area">
			<div class="wpo-testimonial-active owl-carousel">
				<?php 	// Group Param Output
				if (is_array($testimonialItems_groups) && !empty($testimonialItems_groups)) {
					foreach ($testimonialItems_groups as $each_items) {

						$testimonial_title = !empty($each_items['testimonial_title']) ? $each_items['testimonial_title'] : '';
						$testimonial_subtitle = !empty($each_items['testimonial_subtitle']) ? $each_items['testimonial_subtitle'] : '';
						$testimonial_content = !empty($each_items['testimonial_content']) ? $each_items['testimonial_content'] : '';
						$image_url = wp_get_attachment_url($each_items['bg_image']['id']);
						$image_alt = get_post_meta($each_items['bg_image']['id'], '_wp_attachment_image_alt', true);

						$rating_url = wp_get_attachment_url($each_items['rating_image']['id']);
						$rating_alt = get_post_meta($each_items['rating_image']['id'], '_wp_attachment_image_alt', true);

				?>
						<div class="wpo-testimonial-wrap">
							<div class="wpo-testimonial-item">
								<div class="wpo-testimonial-sub">
									<div class="wpo-testimonial-img">
										<?php
										if ($image_url) {
											echo '<img src="' . esc_url($image_url) . '" alt="' . esc_attr($image_alt) . '">';
										}
										?>
									</div>
									<div class="testimonial-tumb-content">
										<?php
										if ($rating_url) {
											echo '<img src="' . esc_url($rating_url) . '" alt="' . esc_attr($rating_alt) . '">';
										}
										if ($testimonial_title) {
											echo '<h4>' . esc_html($testimonial_title) . '<span>' . esc_html($testimonial_subtitle) . '</span></h4>';
										}
										?>
									</div>
								</div>
								<div class="wpo-testimonial-content">
									<?php if ($testimonial_content) {
										echo '<p>' . esc_html($testimonial_content) . '</p>';
									} ?>
								</div>
							</div>
						</div>
				<?php }
				} ?>
			</div>
		</div>
<?php
		// Return outbut buffer
		echo ob_get_clean();
	}
	/**
	 * Render Testimonial widget output in the editor.
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 */

	//protected function _content_template(){}

}
Plugin::instance()->widgets_manager->register(new Apeix_Testimonial());
