<?php
/*
 * Elementor Apeix Project Widget
 * Author & Copyright: wpoceans
*/

namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

class Apeix_Project extends Widget_Base
{

	/**
	 * Retrieve the widget name.
	 */
	public function get_name()
	{
		return 'tmx-apeix_project';
	}

	/**
	 * Retrieve the widget title.
	 */
	public function get_title()
	{
		return esc_html__('Project', 'apeix-core');
	}

	/**
	 * Retrieve the widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-gallery-grid';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 */
	public function get_categories()
	{
		return ['wpoceans-category'];
	}

	/**
	 * Retrieve the list of scripts the Apeix Project widget depended on.
	 * Used to set scripts dependencies required to run the widget.
	 */
	public function get_script_depends()
	{
		return ['tmx-apeix_project'];
	}

	/**
	 * Register Apeix Project widget controls.
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 */
	protected function register_controls()
	{


		$posts = get_posts('post_type="project"&numberposts=-1');
		$PostID = array();
		if ($posts) {
			foreach ($posts as $post) {
				$PostID[$post->ID] = $post->ID;
			}
		} else {
			$PostID[__('No ID\'s found', 'apeix')] = 0;
		}


		$this->start_controls_section(
			'section_project_listing',
			[
				'label' => esc_html__('Listing Options', 'apeix-core'),
			]
		);
		$this->add_control(
			'project_limit',
			[
				'label' => esc_html__('Project Limit', 'apeix-core'),
				'type' => Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 100,
				'step' => 1,
				'default' => 3,
				'description' => esc_html__('Enter the number of items to show.', 'apeix-core'),
			]
		);
		$this->add_control(
			'project_order',
			[
				'label' => __('Order', 'apeix-core'),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'ASC' => esc_html__('Asending', 'apeix-core'),
					'DESC' => esc_html__('Desending', 'apeix-core'),
				],
				'default' => 'DESC',
			]
		);
		$this->add_control(
			'project_orderby',
			[
				'label' => __('Order By', 'apeix-core'),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'none' => esc_html__('None', 'apeix-core'),
					'ID' => esc_html__('ID', 'apeix-core'),
					'author' => esc_html__('Author', 'apeix-core'),
					'title' => esc_html__('Title', 'apeix-core'),
					'date' => esc_html__('Date', 'apeix-core'),
				],
				'default' => 'date',
			]
		);
		$this->add_control(
			'project_show_category',
			[
				'label' => __('Certain Categories?', 'apeix-core'),
				'type' => Controls_Manager::SELECT2,
				'default' => [],
				'options' => Controls_Helper_Output::get_terms_names('project_category'),
				'multiple' => true,
			]
		);
		$this->add_control(
			'project_show_id',
			[
				'label' => __('Certain ID\'s?', 'apeix-core'),
				'type' => Controls_Manager::SELECT2,
				'default' => [],
				'options' => $PostID,
				'multiple' => true,
			]
		);
		$this->end_controls_section(); // end: Section

		// Project Item
		$this->start_controls_section(
			'section_item_overly_style',
			[
				'label' => esc_html__('Overly', 'apeix-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'item_wrapper_bg_color',
			[
				'label' => esc_html__('Background Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .project-section .project-grids .project-content' => 'background-color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'item_wrapper_bg_hover_color',
			[
				'label' => esc_html__('Hover Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .project-section .project-grids .project-content:hover' => 'background-color: {{VALUE}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section

		// Subtitle
		$this->start_controls_section(
			'section_subtitle_style',
			[
				'label' => esc_html__('Subtitle', 'apeix-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'subtitle_color',
			[
				'label' => esc_html__('Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .project-section .project-grids .project-content p' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'subtitle_padding',
			[
				'label' => esc_html__('Subtitle Padding', 'apeix-core'),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em'],
				'selectors' => [
					'{{WRAPPER}} .project-section .project-grids .project-content p' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section

		// Title
		$this->start_controls_section(
			'section_title_style',
			[
				'label' => esc_html__('Title', 'apeix-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'apeix-core'),
				'name' => 'apeix_title_typography',
				'selector' => '{{WRAPPER}} .project-section .project-grids .project-content h2',
			]
		);
		$this->add_control(
			'title_color',
			[
				'label' => esc_html__('Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .project-section .project-grids .project-content h2 a' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'title_padding',
			[
				'label' => esc_html__('Title Padding', 'apeix-core'),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em'],
				'selectors' => [
					'{{WRAPPER}} .project-section .project-grids .project-content h2 a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section


	}

	/**
	 * Render Project widget output on the frontend.
	 * Written in PHP and used to generate the final HTML.
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();

		$project_limit = !empty($settings['project_limit']) ? $settings['project_limit'] : '';
		$project_order = !empty($settings['project_order']) ? $settings['project_order'] : '';
		$project_orderby = !empty($settings['project_orderby']) ? $settings['project_orderby'] : '';
		$project_show_category = !empty($settings['project_show_category']) ? $settings['project_show_category'] : [];
		$project_show_id = !empty($settings['project_show_id']) ? $settings['project_show_id'] : [];


		// Turn output buffer on
		ob_start();

		// Pagination
		global $paged;
		if (get_query_var('paged'))
			$my_page = get_query_var('paged');
		else {
			if (get_query_var('page'))
				$my_page = get_query_var('page');
			else
				$my_page = 1;
			set_query_var('paged', $my_page);
			$paged = $my_page;
		}

		if ($project_show_id) {
			$project_show_id = json_encode($project_show_id);
			$project_show_id = str_replace(array('[', ']'), '', $project_show_id);
			$project_show_id = str_replace(array('"', '"'), '', $project_show_id);
			$project_show_id = explode(',', $project_show_id);
		} else {
			$project_show_id = '';
		}

		$args = array(
			// other query params here,
			'paged' => $my_page,
			'post_type' => 'project',
			'posts_per_page' => (int)$project_limit,
			'category_name' => implode(',', $project_show_category),
			'orderby' => $project_orderby,
			'order' => $project_order,
			'post__in' => $project_show_id,
		);

		$apeix_project = new \WP_Query($args);
		if ($apeix_project->have_posts()) :

?>
			<div class="project-section">
				<div class="container">
					<div class="row">
						<div class="project-grids clearfix">
							<?php
							while ($apeix_project->have_posts()) : $apeix_project->the_post();

								$project_options = get_post_meta(get_the_ID(), 'project_options', true);
								$project_title = isset($project_options['project_title']) ? $project_options['project_title'] : '';
								$project_subtitle = isset($project_options['project_subtitle']) ? $project_options['project_subtitle'] : '';
								$project_image = isset($project_options['project_image']) ? $project_options['project_image'] : '';
								global $post;
								$image_url = wp_get_attachment_url($project_image);
								$image_alt = get_post_meta($project_image, '_wp_attachment_image_alt', true);

							?>
								<div class="grid">
									<div class="entry-media">
										<?php if ($image_url) {
											echo '<img src="' . esc_url($image_url) . '" alt="' . esc_attr($image_alt) . '">';
										} ?>
									</div>
									<div class="project-content">
										<h2><a href="<?php echo esc_url(get_permalink()); ?>"><?php echo esc_html($project_title) ?></a></h2>
										<p class="cat"><?php echo esc_html($project_subtitle) ?></p>
									</div>
								</div>
							<?php
							endwhile;
							wp_reset_postdata();
							?>
						</div>
					</div>
				</div>
			</div>
<?php
		endif;
		// Return outbut buffer
		echo ob_get_clean();
	}
	/**
	 * Render Project widget output in the editor.
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 */

	//protected function _content_template(){}

}
Plugin::instance()->widgets_manager->register(new Apeix_Project());
