<?php
/*
 * Elementor Apeix Blog Widget
 * Author & Copyright: wpoceans
*/

namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

class Apeix_Blog extends Widget_Base
{

	/**
	 * Retrieve the widget name.
	 */
	public function get_name()
	{
		return 'tmx-apeix_blog';
	}

	/**
	 * Retrieve the widget title.
	 */
	public function get_title()
	{
		return esc_html__('Blog', 'apeix-core');
	}

	/**
	 * Retrieve the widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-post';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 */
	public function get_categories()
	{
		return ['wpoceans-category'];
	}

	/**
	 * Retrieve the list of scripts the Apeix Blog widget depended on.
	 * Used to set scripts dependencies required to run the widget.
	 */
	public function get_script_depends()
	{
		return ['tmx-apeix_blog'];
	}

	/**
	 * Register Apeix Blog widget controls.
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 */
	protected function register_controls()
	{

		$posts = get_posts('post_type="post"&numberposts=-1');
		$PostID = array();
		if ($posts) {
			foreach ($posts as $post) {
				$PostID[$post->ID] = $post->ID;
			}
		} else {
			$PostID[__('No ID\'s found', 'apeix')] = 0;
		}

		$this->start_controls_section(
			'section_blog_metas',
			[
				'label' => esc_html__('Meta\'s Options', 'apeix-core'),
			]
		);
		$this->add_control(
			'blog_image',
			[
				'label' => esc_html__('Image', 'apeix-core'),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__('Show', 'apeix-core'),
				'label_off' => esc_html__('Hide', 'apeix-core'),
				'return_value' => 'true',
				'default' => 'true',
			]
		);
		$this->add_control(
			'blog_date',
			[
				'label' => esc_html__('Date', 'apeix-core'),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__('Show', 'apeix-core'),
				'label_off' => esc_html__('Hide', 'apeix-core'),
				'return_value' => 'true',
				'default' => 'true',
			]
		);
		$this->add_control(
			'blog_author',
			[
				'label' => esc_html__('Author', 'apeix-core'),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__('Show', 'apeix-core'),
				'label_off' => esc_html__('Hide', 'apeix-core'),
				'return_value' => 'true',
				'default' => 'true',
			]
		);
		$this->add_control(
			'read_more_txt',
			[
				'label' => esc_html__('Read More Text', 'apeix-core'),
				'type' => Controls_Manager::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__('Type your Read More text here', 'apeix-core'),
			]
		);
		$this->end_controls_section(); // end: Section


		$this->start_controls_section(
			'section_blog_listing',
			[
				'label' => esc_html__('Listing Options', 'apeix-core'),
			]
		);
		$this->add_control(
			'blog_limit',
			[
				'label' => esc_html__('Blog Limit', 'apeix-core'),
				'type' => Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 100,
				'step' => 1,
				'default' => 3,
				'description' => esc_html__('Enter the number of items to show.', 'apeix-core'),
			]
		);
		$this->add_control(
			'blog_order',
			[
				'label' => __('Order', 'apeix-core'),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'ASC' => esc_html__('Asending', 'apeix-core'),
					'DESC' => esc_html__('Desending', 'apeix-core'),
				],
				'default' => 'DESC',
			]
		);
		$this->add_control(
			'blog_orderby',
			[
				'label' => __('Order By', 'apeix-core'),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'none' => esc_html__('None', 'apeix-core'),
					'ID' => esc_html__('ID', 'apeix-core'),
					'author' => esc_html__('Author', 'apeix-core'),
					'title' => esc_html__('Title', 'apeix-core'),
					'date' => esc_html__('Date', 'apeix-core'),
				],
				'default' => 'date',
			]
		);
		$this->add_control(
			'blog_show_category',
			[
				'label' => __('Certain Categories?', 'apeix-core'),
				'type' => Controls_Manager::SELECT2,
				'default' => [],
				'options' => Controls_Helper_Output::get_terms_names('category'),
				'multiple' => true,
			]
		);
		$this->add_control(
			'blog_show_id',
			[
				'label' => __('Certain ID\'s?', 'apeix-core'),
				'type' => Controls_Manager::SELECT2,
				'default' => [],
				'options' => $PostID,
				'multiple' => true,
			]
		);
		$this->add_control(
			'short_content',
			[
				'label' => esc_html__('Excerpt Length', 'apeix-core'),
				'type' => Controls_Manager::NUMBER,
				'min' => 1,
				'step' => 1,
				'default' => 55,
				'description' => esc_html__('How many words you want in short content paragraph.', 'apeix-core'),
				'condition' => [
					'blog_style' => array('style-one'),
				],
			]
		);
		$this->add_control(
			'blog_pagination',
			[
				'label' => esc_html__('Pagination', 'apeix-core'),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__('Show', 'apeix-core'),
				'label_off' => esc_html__('Hide', 'apeix-core'),
				'return_value' => 'true',
				'default' => 'true',
			]
		);

		$this->end_controls_section(); // end: Section


		// Blog Title
		$this->start_controls_section(
			'blog_section_title_style',
			[
				'label' => esc_html__('Title ', 'apeix-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'apeix-core'),
				'name' => 'blog_title_typography',
				'selector' => '{{WRAPPER}} .blog-section .grid .details h3',
			]
		);
		$this->add_control(
			'blog_title_color',
			[
				'label' => esc_html__('Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .blog-section .grid .details h3 a' => 'color: {{VALUE}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section


		// Meta
		$this->start_controls_section(
			'section_meta_style',
			[
				'label' => esc_html__('Meta', 'apeix-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'apeix-core'),
				'name' => 'blog_meta_typography',
				'selector' => '{{WRAPPER}} .blog-section .entry-meta ul li',
			]
		);
		$this->add_control(
			'meta_color',
			[
				'label' => esc_html__('Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .blog-section .entry-meta li,.blog-section .entry-meta li a, .blog-section .entry-meta li span' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'meta_padding',
			[
				'label' => __('Title Padding', 'apeix-core'),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em'],
				'selectors' => [
					'{{WRAPPER}} .blog-section .entry-meta li' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section


	}

	/**
	 * Render Blog widget output on the frontend.
	 * Written in PHP and used to generate the final HTML.
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();
		$read_more_txt = !empty($settings['read_more_txt']) ? $settings['read_more_txt'] : '';
		$blog_limit = !empty($settings['blog_limit']) ? $settings['blog_limit'] : '';
		$blog_image  = (isset($settings['blog_image']) && ('true' == $settings['blog_image'])) ? true : false;
		$blog_date  = (isset($settings['blog_date']) && ('true' == $settings['blog_date'])) ? true : false;
		$blog_author  = (isset($settings['blog_author']) && ('true' == $settings['blog_author'])) ? true : false;

		$blog_order = !empty($settings['blog_order']) ? $settings['blog_order'] : '';
		$blog_orderby = !empty($settings['blog_orderby']) ? $settings['blog_orderby'] : '';
		$blog_show_category = !empty($settings['blog_show_category']) ? $settings['blog_show_category'] : [];
		$blog_show_id = !empty($settings['blog_show_id']) ? $settings['blog_show_id'] : [];
		$short_content = !empty($settings['short_content']) ? $settings['short_content'] : '';
		$blog_pagination  = (isset($settings['blog_pagination']) && ('true' == $settings['blog_pagination'])) ? true : false;

		$excerpt_length = $short_content ? $short_content : '55';
		$read_more_txt = $read_more_txt ? $read_more_txt : esc_html__('Read More', 'apeix-core');

		// Turn output buffer on
		ob_start();


		// Pagination
		global $paged;
		if (get_query_var('paged'))
			$my_page = get_query_var('paged');
		else {
			if (get_query_var('page'))
				$my_page = get_query_var('page');
			else
				$my_page = 1;
			set_query_var('paged', $my_page);
			$paged = $my_page;
		}

		if ($blog_show_id) {
			$blog_show_id = json_encode($blog_show_id);
			$blog_show_id = str_replace(array('[', ']'), '', $blog_show_id);
			$blog_show_id = str_replace(array('"', '"'), '', $blog_show_id);
			$blog_show_id = explode(',', $blog_show_id);
		} else {
			$blog_show_id = '';
		}

		$args = array(
			// other query params here,
			'paged' => $my_page,
			'post_type' => 'post',
			'posts_per_page' => (int)$blog_limit,
			'category_name' => implode(',', $blog_show_category),
			'orderby' => $blog_orderby,
			'order' => $blog_order,
			'post__in' => $blog_show_id,
		);

		$apeix_post = new \WP_Query($args); ?>

		<div class="apeix-blog blog-section">
			<div class="blog-grids clearfix">
				<?php
				if ($apeix_post->have_posts()) : while ($apeix_post->have_posts()) : $apeix_post->the_post();
						$post_options = get_post_meta(get_the_ID(), 'post_options', true);
						$grid_image = isset($post_options['grid_image']) ? $post_options['grid_image'] : '';
						$image_url = wp_get_attachment_url($grid_image);
						$image_alt = get_post_meta($grid_image, '_wp_attachment_image_alt', true);
				?>
						<div class="grid">
							<div class="entry-media">
								<?php if ($image_url) { ?>
									<img src="<?php echo esc_url($image_url); ?>" alt="<?php echo esc_attr($image_alt); ?>">
								<?php } ?>
							</div>
							<div class="details">
								<h3><a href="<?php echo esc_url(get_permalink()); ?>"><?php echo get_the_title(); ?></a></h3>
								<ul class="entry-meta">
									<li>
										<?php if ($blog_author) { // Author Hide
											echo get_avatar(get_the_author_meta('ID'), 125);
											printf(
												'<span class="author">' . esc_html__(' By: ', 'apeix') . ' </span><a href="%1$s" rel="author">%2$s</a>',
												esc_url(get_author_posts_url(get_the_author_meta('ID'))),
												get_the_author()
											);
										} ?>
									</li>
									<?php if ($blog_date) { ?>
										<li>
											<i class="ti-time"></i>
											<?php echo get_the_date(); ?>
										</li>
									<?php } ?>
								</ul>
							</div>
						</div>
					<?php
					endwhile;
				endif;
				wp_reset_postdata();
				if ($blog_pagination) { ?>
					<div class="page-pagination-wrap">
						<?php echo '<div class="paginations">';
						$big = 999999999;
						echo paginate_links(array(
							'base'      => str_replace($big, '%#%', get_pagenum_link($big)),
							'format'    => '?paged=%#%',
							'total'     => $apeix_post->max_num_pages,
							'show_all'  => false,
							'current'   => max(1, $my_page),
							'prev_text'    => '<div class="ti-arrow-left"></div>',
							'next_text'    => '<div class="ti-arrow-right"></div>',
							'mid_size'  => 1,
							'type'      => 'list'
						));
						echo '</div>'; ?>
					</div>
				<?php } ?>
			</div>
		</div>
<?php
		// Return outbut buffer
		echo ob_get_clean();
	}
	/**
	 * Render Blog widget output in the editor.
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 */

	//protected function _content_template(){}

}
Plugin::instance()->widgets_manager->register(new Apeix_Blog());
