<?php
/*
 * Elementor Apeix Blog Widget
 * Author & Copyright: wpoceans
*/

namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

class Apeix_Slider extends Widget_Base
{

	/**
	 * Retrieve the widget name.
	 */
	public function get_name()
	{
		return 'tmx-apeix_slider';
	}

	/**
	 * Retrieve the widget title.
	 */
	public function get_title()
	{
		return esc_html__('Slider', 'apeix-core');
	}

	/**
	 * Retrieve the widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-slides';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 */
	public function get_categories()
	{
		return ['wpoceans-category'];
	}

	/**
	 * Retrieve the list of scripts the Apeix Slider widget depended on.
	 * Used to set scripts dependencies required to run the widget.
	 */

	public function get_script_depends()
	{
		return ['tmx-apeix_slider'];
	}


	/**
	 * Register Apeix Slider widget controls.
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 */
	protected function register_controls()
	{

		$this->start_controls_section(
			'section_slider',
			[
				'label' => __('Slider Options', 'apeix-core'),
			]
		);
		$this->add_control(
			'slide_style',
			[
				'label' => esc_html__('Slide Style', 'apeix-core'),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'style-one' => esc_html__('Style One', 'apeix-core'),
					'style-two' => esc_html__('Style Two', 'apeix-core'),
					'style-three' => esc_html__('Style Three', 'apeix-core'),
				],
				'default' => 'style-one',
				'description' => esc_html__('Select your Slide style.', 'apeix-core'),
			]
		);
		$repeater = new Repeater();
		$repeater->add_control(
			'slide_color',
			[
				'label' => esc_html__('Slide background Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'default' => '#bbbbbb',
				'selectors' => [
					'{{WRAPPER}} .hero-slider' => 'background-color: {{VALUE}};',
				],
			]
		);
		$repeater->add_control(
			'slider_image',
			[
				'label' => esc_html__('Slider Image', 'apeix-core'),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
			]
		);
		$repeater->add_control(
			'slider_title',
			[
				'label' => esc_html__('Slider title', 'apeix-core'),
				'type' => Controls_Manager::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__('Type slide title here', 'apeix-core'),
			]
		);
		$repeater->add_control(
			'slider_content',
			[
				'label' => esc_html__('Slider content', 'apeix-core'),
				'type' => Controls_Manager::TEXTAREA,
				'label_block' => true,
				'placeholder' => esc_html__('Type slide content here', 'apeix-core'),
			]
		);
		$repeater->add_control(
			'btn_txt',
			[
				'label' => esc_html__('Button Text', 'apeix-core'),
				'type' => Controls_Manager::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__('Type your button text here', 'apeix-core'),
			]
		);
		$repeater->add_control(
			'button_link',
			[
				'label' => esc_html__('Button Link', 'apeix-core'),
				'type' => Controls_Manager::URL,
				'placeholder' => __('https://your-link.com', 'apeix-core'),
				'show_external' => true,
				'default' => [
					'url' => '',
				],
			]
		);
		$repeater->add_control(
			'video_link',
			[
				'label' => esc_html__('Video Button Link', 'apeix-core'),
				'type' => Controls_Manager::TEXT,
				'label_block' => true,
				'default' => 'https://www.youtube.com/embed/7e90gBu4pas?autoplay=1',
				'placeholder' => esc_html__('Type your button link here', 'apeix-core'),
			]
		);
		$this->add_control(
			'swipeSliders_groups',
			[
				'label' => esc_html__('Slider Items', 'apeix-core'),
				'type' => Controls_Manager::REPEATER,
				'default' => [
					[
						'slider_title' => esc_html__('Item #1', 'apeix-core'),
					],

				],
				'fields' =>  $repeater->get_controls(),
				'title_field' => '{{{ slider_title }}}',
			]
		);
		$this->end_controls_section(); // end: Section

		$this->start_controls_section(
			'section_carousel',
			[
				'label' => esc_html__('Carousel Options', 'apeix-core'),
			]
		);
		$this->add_control(
			'carousel_nav',
			[
				'label' => esc_html__('Navigation', 'apeix-core'),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__('Yes', 'apeix-core'),
				'label_off' => esc_html__('No', 'apeix-core'),
				'return_value' => 'true',
				'description' => esc_html__('If you want Carousel Navigation, enable it.', 'apeix-core'),
			]
		);

		$this->end_controls_section(); // end: Section


		// Title
		$this->start_controls_section(
			'section_title_style',
			[
				'label' => esc_html__('Title', 'apeix-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'title_typography',
				'selector' => '{{WRAPPER}} .hero-slider .slide-title h2 ',
			]
		);
		$this->add_control(
			'title_color',
			[
				'label' => esc_html__('Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .hero-slider .slide-title h2' => 'color: {{VALUE}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section

		// Content
		$this->start_controls_section(
			'section_content_style',
			[
				'label' => esc_html__('Content', 'apeix-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'slider_content_typography',
				'selector' => '{{WRAPPER}} .hero-slider .slide-text p',
			]
		);
		$this->add_control(
			'slider_content_color',
			[
				'label' => esc_html__('Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .hero-slider .slide-text p' => 'color: {{VALUE}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section


		// Button Style
		$this->start_controls_section(
			'section_button_style',
			[
				'label' => esc_html__('Button', 'apeix-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'button_typography',
				'selector' => '{{WRAPPER}} .hero-slider .slide-btns .theme-btn',
			]
		);
		$this->add_responsive_control(
			'button_min_width',
			[
				'label' => esc_html__('Width', 'apeix-core'),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 10,
						'max' => 500,
						'step' => 1,
					],
				],
				'size_units' => ['px'],
				'selectors' => [
					'{{WRAPPER}} .hero-slider .slide-btns .theme-btn' => 'min-width: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->add_control(
			'button_padding',
			[
				'label' => __('Padding', 'apeix-core'),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em'],
				'selectors' => [
					'{{WRAPPER}} .hero-slider .slide-btns .theme-btn' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_control(
			'button_border_radius',
			[
				'label' => __('Border Radius', 'apeix-core'),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em'],
				'selectors' => [
					'{{WRAPPER}} .hero-slider .slide-btns .theme-btn' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->start_controls_tabs('button_style');
		$this->start_controls_tab(
			'button_normal',
			[
				'label' => esc_html__('Normal', 'apeix-core'),
			]
		);
		$this->add_control(
			'button_color',
			[
				'label' => esc_html__('Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .hero-slider .slide-btns .theme-btn' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name' => 'button_bg_color',
				'label' => esc_html__('Heading BG Color', 'apeix-core'),
				'types' => ['gradient'],
				'selector' => '{{WRAPPER}} .hero-slider .slide-btns .theme-btn',
			]
		);
		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'button_border',
				'label' => esc_html__('Border', 'apeix-core'),
				'selector' => '{{WRAPPER}} .hero-slider .slide-btns .theme-btn',
			]
		);
		$this->end_controls_tab();  // end:Normal tab

		$this->start_controls_tab(
			'button_hover',
			[
				'label' => esc_html__('Hover', 'apeix-core'),
			]
		);
		$this->add_control(
			'button_hover_color',
			[
				'label' => esc_html__('Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .hero-slider .slide-btns .theme-btn:hover' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name' => 'button_bg_hover_color',
				'label' => esc_html__('Hover BG Color', 'apeix-core'),
				'types' => ['gradient'],
				'selector' => '{{WRAPPER}} .hero-slider .slide-btns .theme-btn:hover',
			]
		);
		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'button_hover_border',
				'label' => esc_html__('Border', 'apeix-core'),
				'selector' => '{{WRAPPER}} .hero-slider .slide-btns .theme-btn:hover',
			]
		);
		$this->end_controls_tab();  // end:Hover tab
		$this->end_controls_tabs(); // end tabs

		$this->end_controls_section(); // end: Section

		// Navigation
		$this->start_controls_section(
			'section_navigation_style',
			[
				'label' => esc_html__('Navigation', 'apeix-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'slider_nav_color',
			[
				'label' => esc_html__('Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .hero-slider .swiper-button-prev:before,.hero-slider .swiper-button-next:before' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'slider_nav_bg_color',
			[
				'label' => esc_html__('Background Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .hero-slider .swiper-button-prev, .hero-slider .swiper-button-next' => 'background-color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'slider_nav_border_color',
			[
				'label' => esc_html__('Border Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .hero-slider .swiper-button-prev, .hero-slider .swiper-button-next' => 'border-color: {{VALUE}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section

		// Video 
		$this->start_controls_section(
			'section_video_btn_style',
			[
				'label' => esc_html__('Video', 'apeix-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'video_bg',
			[
				'label' => esc_html__('BG Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .slide-video-btn a.video-btn' => 'background-color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'video_icon_bg',
			[
				'label' => esc_html__('Icon Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .slide-video-btn a.video-btn:before' => 'border-left-color: {{VALUE}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section


	}

	/**
	 * Render Blog widget output on the frontend.
	 * Written in PHP and used to generate the final HTML.
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();

		// Carousel Options
		$swipeSliders_groups = !empty($settings['swipeSliders_groups']) ? $settings['swipeSliders_groups'] : [];
		$carousel_nav  = (isset($settings['carousel_nav']) && ('true' == $settings['carousel_nav'])) ? true : false;
		$slide_style = !empty($settings['slide_style']) ? $settings['slide_style'] : '';

		if ($slide_style == 'style-one') {
			$hero_style = 'hero-style-2 hero-style-1';
		} elseif ($slide_style == 'style-two') {
			$hero_style = 'hero-style-2';
		} else {
			$hero_style = 'hero-style-3';
		}

		// Turn output buffer on
		ob_start();

?>
		<div class="hero-slider <?php echo esc_attr($hero_style); ?>">
			<div class="swiper-container">
				<div class="swiper-wrapper">
					<?php
					if (is_array($swipeSliders_groups) && !empty($swipeSliders_groups)) {
						foreach ($swipeSliders_groups as $each_item) {

							$bg_image = !empty($each_item['slider_image']['id']) ? $each_item['slider_image']['id'] : '';
							$image_url = wp_get_attachment_url($bg_image);
							$slider_title = !empty($each_item['slider_title']) ? $each_item['slider_title'] : '';
							$slider_content = !empty($each_item['slider_content']) ? $each_item['slider_content'] : '';

							$video_link = !empty($each_item['video_link']) ? $each_item['video_link'] : '';

							$button_text = !empty($each_item['btn_txt']) ? $each_item['btn_txt'] : '';
							$button_link = !empty($each_item['button_link']['url']) ? $each_item['button_link']['url'] : '';
							$button_link_external = !empty($each_item['button_link']['is_external']) ? 'target="_blank"' : '';
							$button_link_nofollow = !empty($each_item['button_link']['nofollow']) ? 'rel="nofollow"' : '';
							$button_link_attr = !empty($button_link) ?  $button_link_external . ' ' . $button_link_nofollow : '';


							$button_one = $button_link ? '<a href="' . esc_url($button_link) . '" ' . $button_link_attr . ' class="theme-btn wpo-theme-btn color-9" >' . $button_text . '</a>' : '';

							$button_actual = ($button_one) ? '<div data-swiper-parallax="500" class="slide-btns">' . $button_one . '</div>' : '';

					?>
							<div class="swiper-slide">
								<div class="slide-inner slide-bg-image" data-background="<?php echo esc_url($image_url); ?>">
									<div class="gradient-overlay"></div>
									<div class="container">
										<?php
										if ($slider_title) { ?>
											<div data-swiper-parallax="300" class="slide-title">
												<h2><?php echo esc_html($slider_title); ?></h2>
											</div>
										<?php }
										if ($slider_content) { ?>
											<div data-swiper-parallax="400" class="slide-text">
												<p><?php echo esc_html($slider_content); ?></p>
											</div>
										<?php } ?>
										<div class="clearfix"></div>
										<div class="btn-wrapper">
											<?php echo $button_actual; ?>

											<?php if ($video_link) { ?>
												<div class="slide-video-btn">
													<a href="<?php echo esc_url($video_link); ?>" class="video-btn" data-type="iframe"></a>
												</div>
											<?php } ?>

										</div>

									</div>
								</div> <!-- end slide-inner -->
							</div><!-- end swiper-slide -->
					<?php }
					} ?>
				</div>
				<!-- swipper controls -->
				<?php if ($carousel_nav) { ?>
					<div class="swiper-pagination"></div>
					<div class="swiper-button-next"></div>
					<div class="swiper-button-prev"></div>
				<?php } ?>
			</div>
		</div>
<?php
		// Return outbut buffer
		echo ob_get_clean();
	}

	/**
	 * Render Blog widget output in the editor.
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 */

	//protected function _content_template(){}

}
Plugin::instance()->widgets_manager->register(new Apeix_Slider());
