<?php
/*
 * Elementor Apeix Service Widget
 * Author & Copyright: wpoceans
*/

namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

class Apeix_Service extends Widget_Base
{

	/**
	 * Retrieve the widget name.
	 */
	public function get_name()
	{
		return 'tmx-apeix_service';
	}

	/**
	 * Retrieve the widget title.
	 */
	public function get_title()
	{
		return esc_html__('Service', 'apeix-core');
	}

	/**
	 * Retrieve the widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-archive';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 */
	public function get_categories()
	{
		return ['wpoceans-category'];
	}

	/**
	 * Retrieve the list of scripts the Apeix Service widget depended on.
	 * Used to set scripts dependencies required to run the widget.
	 */
	public function get_script_depends()
	{
		return ['tmx-apeix_service'];
	}

	/**
	 * Register Apeix Service widget controls.
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 */
	protected function register_controls()
	{


		$posts = get_posts('post_type="service"&numberposts=-1');
		$PostID = array();
		if ($posts) {
			foreach ($posts as $post) {
				$PostID[$post->ID] = $post->ID;
			}
		} else {
			$PostID[__('No ID\'s found', 'apeix')] = 0;
		}


		$this->start_controls_section(
			'section_service_listing',
			[
				'label' => esc_html__('Listing Options', 'apeix-core'),
			]
		);
		$this->add_control(
			'service_style',
			[
				'label' => esc_html__('Service Style', 'annahl-core'),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'style-one' => esc_html__('Style One', 'annahl-core'),
					'style-two' => esc_html__('Style Two', 'annahl-core'),
				],
				'default' => 'style-one',
				'description' => esc_html__('Select your service style.', 'annahl-core'),
			]
		);
		$this->add_control(
			'service_limit',
			[
				'label' => esc_html__('Service Limit', 'apeix-core'),
				'type' => Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 100,
				'step' => 1,
				'default' => 3,
				'description' => esc_html__('Enter the number of items to show.', 'apeix-core'),
			]
		);
		$this->add_control(
			'short_content',
			[
				'label' => esc_html__('Excerpt Length', 'apeix-core'),
				'type' => Controls_Manager::NUMBER,
				'min' => 1,
				'step' => 1,
				'default' => 17,
				'description' => esc_html__('How many words you want in short content paragraph.', 'apeix-core'),
			]
		);
		$this->add_control(
			'service_order',
			[
				'label' => __('Order', 'apeix-core'),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'ASC' => esc_html__('Asending', 'apeix-core'),
					'DESC' => esc_html__('Desending', 'apeix-core'),
				],
				'default' => 'DESC',
			]
		);
		$this->add_control(
			'service_orderby',
			[
				'label' => __('Order By', 'apeix-core'),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'none' => esc_html__('None', 'apeix-core'),
					'ID' => esc_html__('ID', 'apeix-core'),
					'author' => esc_html__('Author', 'apeix-core'),
					'title' => esc_html__('Title', 'apeix-core'),
					'date' => esc_html__('Date', 'apeix-core'),
				],
				'default' => 'date',
			]
		);
		$this->add_control(
			'service_show_category',
			[
				'label' => __('Certain Categories?', 'apeix-core'),
				'type' => Controls_Manager::SELECT2,
				'default' => [],
				'options' => Controls_Helper_Output::get_terms_names('service_category'),
				'multiple' => true,
			]
		);
		$this->add_control(
			'service_show_id',
			[
				'label' => __('Certain ID\'s?', 'apeix-core'),
				'type' => Controls_Manager::SELECT2,
				'default' => [],
				'options' => $PostID,
				'multiple' => true,
			]
		);
		$this->end_controls_section(); // end: Section

		// Section
		$this->start_controls_section(
			'service_section_icon_style',
			[
				'label' => esc_html__('Icon', 'apeix-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'apeix-core'),
				'name' => 'service_apeix_icon_typography',
				'selector' => '{{WRAPPER}} .apeix-services .service-icon:before',
			]
		);
		$this->add_control(
			'service_icon_color',
			[
				'label' => esc_html__('Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .apeix-services .service-icon:before' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'service_number_color',
			[
				'label' => esc_html__('Number Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .apeix-services .service-icon:before' => 'color: {{VALUE}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section

		// Title
		$this->start_controls_section(
			'service_section_title_style',
			[
				'label' => esc_html__('Title', 'apeix-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'apeix-core'),
				'name' => 'service_apeix_title_typography',
				'selector' => '{{WRAPPER}} .apeix-services .service-text h2',
			]
		);
		$this->add_control(
			'service_section_title_color',
			[
				'label' => esc_html__('Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .apeix-services .service-text h2 a' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'service_section_title_padding',
			[
				'label' => esc_html__('Title Padding', 'apeix-core'),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em'],
				'selectors' => [
					'{{WRAPPER}} .apeix-services .service-text h2' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section

		// Content
		$this->start_controls_section(
			'service_section_content_style',
			[
				'label' => esc_html__('Content', 'apeix-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'apeix-core'),
				'name' => 'service_apeix_content_typography',
				'selector' => '{{WRAPPER}} .apeix-services .service-text p',
			]
		);
		$this->add_control(
			'service_section_content_color',
			[
				'label' => esc_html__('Color', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .apeix-services .service-text p' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'service_section_content_padding',
			[
				'label' => esc_html__('Title Padding', 'apeix-core'),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em'],
				'selectors' => [
					'{{WRAPPER}} .apeix-services .service-text p' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section

		// Content 
		$this->start_controls_section(
			'service_section_content_bg_style',
			[
				'label' => esc_html__('Content', 'apeix-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'service_content_bg_color',
			[
				'label' => esc_html__('Content BG', 'apeix-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ..apeix-services .service-wrap' => 'color: {{VALUE}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section


	}

	/**
	 * Render Service widget output on the frontend.
	 * Written in PHP and used to generate the final HTML.
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();
		$service_style = !empty($settings['service_style']) ? $settings['service_style'] : '';
		$service_limit = !empty($settings['service_limit']) ? $settings['service_limit'] : '';
		$service_order = !empty($settings['service_order']) ? $settings['service_order'] : '';
		$service_orderby = !empty($settings['service_orderby']) ? $settings['service_orderby'] : '';
		$service_show_category = !empty($settings['service_show_category']) ? $settings['service_show_category'] : [];
		$service_show_id = !empty($settings['service_show_id']) ? $settings['service_show_id'] : [];
		$short_content = !empty($settings['short_content']) ? $settings['short_content'] : '';
		$excerpt_length = $short_content ? $short_content : '17';

		if ($service_style == 'style-one') {
			$wrapper_class = 'service-style-1';
		} else {
			$wrapper_class = 'service-style-3';
		}

		// Turn output buffer on
		ob_start();

		// Pagination
		global $paged;
		if (get_query_var('paged'))
			$my_page = get_query_var('paged');
		else {
			if (get_query_var('page'))
				$my_page = get_query_var('page');
			else
				$my_page = 1;
			set_query_var('paged', $my_page);
			$paged = $my_page;
		}

		if ($service_show_id) {
			$service_show_id = json_encode($service_show_id);
			$service_show_id = str_replace(array('[', ']'), '', $service_show_id);
			$service_show_id = str_replace(array('"', '"'), '', $service_show_id);
			$service_show_id = explode(',', $service_show_id);
		} else {
			$service_show_id = '';
		}

		$args = array(
			// other query params here,
			'paged' => $my_page,
			'post_type' => 'service',
			'posts_per_page' => (int)$service_limit,
			'category_name' => implode(',', $service_show_category),
			'orderby' => $service_orderby,
			'order' => $service_order,
			'post__in' => $service_show_id,
		);

		$apeix_service = new \WP_Query($args);
?>
		<div class="apeix-services <?php echo esc_attr($wrapper_class); ?> ">
			<div class="container">
				<div class="row">
					<?php
					if ($apeix_service->have_posts()) : while ($apeix_service->have_posts()) : $apeix_service->the_post();
							$service_options = get_post_meta(get_the_ID(), 'service_options', true);
							$service_title = isset($service_options['service_title']) ? $service_options['service_title'] : '';
							$service_icon = isset($service_options['service_icon']) ? $service_options['service_icon'] : '';
							$service_count = isset($service_options['service_count']) ? $service_options['service_count'] : '';
					?>
							<div class="col-md-4 col-sm-6 sgrid">
								<div class="service-wrap">
									<?php if ($service_icon) { ?>
										<div class="service-icon">
											<i class="fi <?php echo esc_attr($service_icon); ?>"></i>
										</div>
									<?php	} ?>
									<div class="service-text">
										<h2><a href="<?php echo esc_url(get_permalink()); ?>">
												<?php echo esc_html($service_title); ?></a>
										</h2>
										<p><?php echo wp_trim_words(get_the_excerpt(), $excerpt_length, ' '); ?></p>
									</div>
								</div>
							</div>
					<?php
						endwhile;
					endif;
					wp_reset_postdata();
					?>
				</div>
			</div>
		</div>
<?php
		// Return outbut buffer
		echo ob_get_clean();
	}
	/**
	 * Render Service widget output in the editor.
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 */

	//protected function _content_template(){}

}
Plugin::instance()->widgets_manager->register(new Apeix_Service());
